﻿# PowerShell Script to Configure TLS Settings on Windows
# This script disables SSL 2.0, SSL 3.0, TLS 1.0, and TLS 1.1
# and ensures TLS 1.2 is enabled for both client and server

# Check if running as Administrator
if (-NOT ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole] "Administrator")) {
    Write-Error "This script requires Administrator privileges. Please run PowerShell as Administrator."
    exit 1
}

Write-Host "Configuring TLS settings..." -ForegroundColor Green

# Function to create registry keys if they don't exist
function New-RegistryKeyIfNotExists {
    param(
        [string]$Path
    )
    try {
        if (!(Test-Path $Path)) {
            New-Item -Path $Path -Force | Out-Null
            Write-Host "Created registry key: $Path" -ForegroundColor Yellow
        }
    }
    catch {
        Write-Error "Failed to create registry key: $Path. Error: $($_.Exception.Message)"
        return $false
    }
    return $true
}

# Function to set registry values for protocol configuration
function Set-ProtocolRegistry {
    param(
        [string]$Protocol,
        [string]$Type,  # "Client" or "Server"
        [int]$Enabled,  # 0 = Disabled, 1 = Enabled
        [int]$DisabledByDefault  # 0 = Not disabled by default, 1 = Disabled by default
    )
    
    $registryPath = "HKLM:\SYSTEM\CurrentControlSet\Control\SecurityProviders\SCHANNEL\Protocols\$Protocol\$Type"
    
    try {
        # Create registry key if it doesn't exist
        if (!(New-RegistryKeyIfNotExists -Path $registryPath)) {
            return $false
        }
        
        # Set registry values
        Set-ItemProperty -Path $registryPath -Name "Enabled" -Value $Enabled -Type DWord -ErrorAction Stop
        Set-ItemProperty -Path $registryPath -Name "DisabledByDefault" -Value $DisabledByDefault -Type DWord -ErrorAction Stop
        
        $status = if ($Enabled -eq 0) { "DISABLED" } else { "ENABLED" }
        Write-Host "$Protocol $Type : $status" -ForegroundColor $(if ($status -eq "DISABLED") { "Red" } else { "Green" })
        return $true
    }
    catch {
        Write-Error "Failed to configure $Protocol $Type. Error: $($_.Exception.Message)"
        return $false
    }
}

Write-Host "`nDisabling insecure protocols..." -ForegroundColor Yellow

$success = $true

# Disable SSL 2.0
$success = $success -and (Set-ProtocolRegistry -Protocol "SSL 2.0" -Type "Client" -Enabled 0 -DisabledByDefault 1)
$success = $success -and (Set-ProtocolRegistry -Protocol "SSL 2.0" -Type "Server" -Enabled 0 -DisabledByDefault 1)

# Disable SSL 3.0
$success = $success -and (Set-ProtocolRegistry -Protocol "SSL 3.0" -Type "Client" -Enabled 0 -DisabledByDefault 1)
$success = $success -and (Set-ProtocolRegistry -Protocol "SSL 3.0" -Type "Server" -Enabled 0 -DisabledByDefault 1)

# Disable TLS 1.0
$success = $success -and (Set-ProtocolRegistry -Protocol "TLS 1.0" -Type "Client" -Enabled 0 -DisabledByDefault 1)
$success = $success -and (Set-ProtocolRegistry -Protocol "TLS 1.0" -Type "Server" -Enabled 0 -DisabledByDefault 1)

# Disable TLS 1.1
$success = $success -and (Set-ProtocolRegistry -Protocol "TLS 1.1" -Type "Client" -Enabled 0 -DisabledByDefault 1)
$success = $success -and (Set-ProtocolRegistry -Protocol "TLS 1.1" -Type "Server" -Enabled 0 -DisabledByDefault 1)

Write-Host "`nEnabling secure protocols..." -ForegroundColor Yellow

# Enable TLS 1.2
$success = $success -and (Set-ProtocolRegistry -Protocol "TLS 1.2" -Type "Client" -Enabled 1 -DisabledByDefault 0)
$success = $success -and (Set-ProtocolRegistry -Protocol "TLS 1.2" -Type "Server" -Enabled 1 -DisabledByDefault 0)

if ($success) {
    Write-Host "`nTLS configuration completed successfully!" -ForegroundColor Green
    Write-Host "A system reboot is recommended for all changes to take effect." -ForegroundColor Cyan
} else {
    Write-Host "`nTLS configuration completed with some errors. Please check the output above." -ForegroundColor Red
}

# Optional: Display current configuration
Write-Host "`nCurrent TLS Registry Configuration:" -ForegroundColor Magenta
$protocols = @("SSL 2.0", "SSL 3.0", "TLS 1.0", "TLS 1.1", "TLS 1.2")
$types = @("Client", "Server")

foreach ($protocol in $protocols) {
    foreach ($type in $types) {
        $path = "HKLM:\SYSTEM\CurrentControlSet\Control\SecurityProviders\SCHANNEL\Protocols\$protocol\$type"
        if (Test-Path $path) {
            $enabled = (Get-ItemProperty -Path $path -Name "Enabled" -ErrorAction SilentlyContinue).Enabled
            $disabled = (Get-ItemProperty -Path $path -Name "DisabledByDefault" -ErrorAction SilentlyContinue).DisabledByDefault
            $status = if ($enabled -eq 0 -or $disabled -eq 1) { "DISABLED" } else { "ENABLED" }
            Write-Host "$protocol $type : $status (Enabled=$enabled, DisabledByDefault=$disabled)"
        }
    }
}